=begin
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
# Designed September 2009 by Fredo6

# Permission to use this software for any purpose and without fee is hereby granted
# Distribution of this software for commercial purpose is subject to:
#  - the expressed, written consent of the author
#  - the inclusion of the present copyright notice in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name			:   CurviloftSkinning.rb
# Original Date	:   4 Jan 2010 - version 1.0
# Description	:   Specific method for Skinning in the Curviloft script family
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
=end

module Curviloft

class CVL_LoftAlgo 

#----------------------------------------------------------------------------
# SKINNING: analysis of Contours
#----------------------------------------------------------------------------

#Verify contours for Skin
def skinning_check_contours(lst_cells, from_selection=false)
	#Ordering the plates
	#puts "CELL = #{lst_cells.length} From selection = #{from_selection}"
	#puts " ---> #{lst_cells.inspect}"
	
	#Analysing the cells and creating the plates and links
	@lst_plates = []
	lst_cells.each do |cell|
		#checking if cells has already faces built
		
		#checking if contour is flat
		contour = G6.curl_concat cell
		if G6.points_coplanar?(contour)
			flat = true
		end	
		
		#Building a cell with 4 sides
		rails, pplates = skinning_adjust_cell cell
		#puts "End adjust"
		next unless rails
		
		lplates = pplates.collect { |lpt| plate_create(lpt) }
		@lst_plates += lplates
		#puts "create link skinning"
		link = link_create *lplates
		
		contour = G6.curl_concat cell
		if G6.points_coplanar?(contour)
			flat = true
		end	
		link.flat = flat 
		link.no_edition = flat 
		link.contours = contour
		link.hiborders = [contour]
		link.rails =  G6.curl_harmonize rails[0], rails[1], 0.025
	end

	true
end

#Construct a cell with 4 sides
def skinning_adjust_cell(cell)
	case cell.length
	when 3, 4
		#return cell
	when 1, 2
		cell = skinning_split_cells_by_four(cell)
	else
		return nil
	end	

	#puts "Sepcial Cell #{cell.length}"
	case cell.length
	when 3
		return skinning_split_cells_by_three(cell)
	
	when 4
		n1 = cell[0].length + cell[2].length
		n2 = cell[1].length + cell[3].length
		if n1 > n2
		#if n1 < n2
			pplates = [cell[0], cell[2].reverse]
			rails = [cell[3].reverse, cell[1]]
		else
			pplates = [cell[1], cell[3].reverse]
			rails = [cell[0].reverse, cell[2]]
		end
		####return nil
		
	else
		return nil
	end
	
	#Cell with less parts than 4
	[rails, pplates]
end

def skinning_split_cells_by_three(cell)
	ls = cell.collect { |c| [c, c.length] }
	ls.sort! { |a, b| a[1] <=> b[1] }
	cell = ls.collect { |a| a[0] }
	
	rail1 = cell[0]
	rail2 = cell[1]
	lcom = [rail1.first, rail1.last] & [rail2.first, rail2.last]
	pivot = lcom[0]
	rail1 = rail1.reverse unless rail1.last == pivot
	rail2 = rail2.reverse unless rail2.last == pivot
	#puts "rail1 = #{rail1.length} rail2 = #{rail2.length}"
	rails = [rail1, rail2]
	
	pplate1 = cell[2]
	pplate1 = pplate1.reverse unless pplate1.first == rail1.first
	pplates = [pplate1, lcom]
	
	#puts "pplate1 = #{pplates[0].length} pplate2 = #{pplates[1].length}"
	[rails, pplates]
end

def skinning_split_cells_by_four(cell)
	n = cell.length
	#puts "SPLIT Cell n = #{n}"
	langles = []
	cell.each_with_index do |lpt, icell|
		#puts "cell len = #{lpt.length}"
		for i in 1..lpt.length-2
			pt1 = lpt[i-1]
			pt2 = lpt[i]
			pt3 = lpt[i+1]
			langles.push [icell, i, pt2.vector_to(pt1).angle_between(pt2.vector_to(pt3))]
		end
	end	
	langles = langles.find_all { |a| a[2] > 0 && a[2] < 120.degrees }
	langles = langles.sort { |a, b| a[2] <=> b[2] }
	#puts "SPLIT langles = #{langles.length}"
	#langles.each { |cut| puts "langles = #{cut[1]} - #{cut[2].radians}" }
	
	#nkeep = 4 - n + 1
	nkeep = 4 - n 
	return nil if langles.length < nkeep
	
	cuts = langles[0..nkeep-1]
	#cuts.each { |cut| puts "cut = #{cut[1]} - #{cut[2].radians}" }
	
	#Cutting the cell and rebuiling the new sequence
	newcell = []
	cell.each_with_index do |lpt, icell|
		lipt = (cuts.find_all { |cut| cut[0] == icell }).collect { |a| a[1] }
		#puts "lipt = #{lipt.inspect}"
		newcell += curl_divide_at_index(lpt, *lipt)
	end	
	
	#puts "NEWCELL = #{newcell.length}"
	#puts "Ind = #{(newcell.collect {|c| c.length }).inspect}"
	newcell = newcell[1..-2] + [(newcell[-1] + newcell[0][1..-1])]
	#puts "after NEWCELL = #{newcell.length}"
	#puts "after Ind = #{(newcell.collect {|c| c.length }).inspect}"
	newcell
end

def curl_divide_at_index(curl, *lipt)
	return [curl] if lipt.length == 0
	lipt = lipt.sort { |a, b| a <=> b }
	newcurl = []
	n = curl.length
	ibeg = 0
	lipt.each do |ipt|
		next if ipt >= n
		newcurl.push curl[ibeg..ipt]
		ibeg = ipt
	end
	newcurl.push curl[ibeg..-1]
	newcurl
end

#Get a vertex from a given point
def get_vertex_from_point(pt)
	(@method_get_vertex) ? @method_get_vertex.call(pt) : nil
end

#Get an edge from two consecutive points
def get_edge_from_points(pt1, pt2)
	vx1 = get_vertex_from_point pt1
	return nil unless vx1
	vx2 = get_vertex_from_point pt2
	return nil unless vx2
	vx1.edges.find { |e| e.other_vertex == vx2 }
end

#Construct the Skinning curves for a link (2 pathes)
def skinning_junction_construct(link)	
	rail1 = link.rails[0]
	rail2 = link.rails[1]
	#puts "Junc rail1 = #{rail1.length} rail2 = #{rail2.length}"
	dtot1 = link.tot_len[0]
	dtot2 = link.tot_len[1]
	#puts "dtot1 = #{dtot1} dtot2 = #{dtot2}"
	d1 = 0
	d2 = 0
	ptprev1 = link.pairs[0][0].pt
	ptprev2 = link.pairs[0][1].pt
	lbz_pts = []
	link.pairs.each do |pair|
		pt1 = pair[0].pt
		pt2 = pair[1].pt
		#puts "pt1 = #{pt1} pt2 = #{pt2}"
		d1 += pt1.distance(ptprev1)
		d2 += pt2.distance(ptprev2)
		ptprev1 = pt1
		ptprev2 = pt2
		ratio1 = 1.0 - d1 / dtot1
		ratio2 = (dtot2 == 0) ? 1.0 - ratio1 : d2 / dtot2
		crv1 = G6.curl_move_extremities rail1, pt1, pt2
		crv2 = G6.curl_move_extremities rail2, pt1, pt2
		#puts "CRV1 = #{crv1.length} CRV2 = #{crv2.length}"
		
		sum = ratio1 + ratio2
		#puts "\nbef Ratio1 = #{ratio1} Ratio2 = #{ratio2} sum = #{ratio1 + ratio2}"
		ratio1 = ratio1 / sum
		ratio2 = ratio2 / sum
		#puts "aft Ratio1 = #{ratio1} Ratio2 = #{ratio2} sum = #{ratio1 + ratio2}"
		
		#Computing the variable ratio
		nc = crv1.length - 1
		dc = [0.0]
		dtot = 0.0
		for i in 1..nc
			dtot += crv1[i-1].distance(crv1[i])
			dc[i] = dtot
		end
		
		#Averaging the profiles
		pts = []
		for i in 0..nc
			rc = dc[i] / dtot
			#ratio = 0.5 * ((1.0 - rc) * ratio1 + rc * ratio2)
			rc = 1 - rc
			ratio = (rc * ratio1 + (1.0 - rc) * ratio2)
			#ratio = 1.0 - ratio2
			ratio = 0.5 * (ratio1 + ratio2)
			sum = (ratio1 + ratio2)
			#pts[i] = Geom.linear_combination ratio, crv1[i], 1.0 - ratio, crv2[i]
			#pts[i] = Geom.linear_combination 1-ratio, crv1[i], ratio, crv2[i]
			pt1 = Geom.linear_combination ratio1, crv1[i], (1.0 - ratio1), crv2[i]
			pt2 = Geom.linear_combination (1 - ratio2), crv1[i], ratio2, crv2[i]
			#pt1 = pt2
			pts[i] = Geom.linear_combination 0.5, pt1, 0.5, pt2
		end	
		bz = pts
		#bz = crv2
		lbz_pts.push bz
		#puts "BZ = #{bz.length} #{bz[0]} #{bz.last}"
	end
	lbz_pts
end

end	#End Class CVL_LoftAlgo

end	#End Module Curviloft
